<?php
/** 15.01.2026 Modus list / inline hinzugefügt
 * Plugin:      SimpleCounter-2
 * Version:     3.0.4
 * Datum:       2026-01-22
 * Autor:       hausl78, A. Zimmermann, G. Fischer
 * Lizenz:
 * Für mozilo:  getestet mit moziloCMS 2.x Rev 55
 * PHP:         getestet mit PHP 7.4 / 8.x
 *
 * Kurzbeschreibung:
 *   Besucherzähler, DSGVO-freundlich & barrierefrei
 *   Darstellung als Block oder in Zeile
 */
declare(strict_types=1);

if (!defined('IS_CMS')) {
    die();
}

class SimpleCounter extends Plugin
{
    private const DATA_FILE = 'data.txt';
    private const LANG_DIR  = 'sprachen';

    private string $datafile_rel;
    private string $datafile_abs;

    /** geladene Frontend-Sprachwerte key => text */
    private array $i18n = [];
    /** geladene Admin-Sprachwerte key => text */
    private array $i18nAdmin = [];

    /** aktuell gewählte Sprachdatei, z.B. deDE / enEN */
    private string $locale = 'deDE';

    private bool $langLoaded = false;
    private bool $adminLangLoaded = false;

    /** kleine robuste Default-Map für Frontend-Labels */
    private const DEFAULT_I18N = [
        'deDE' => [
            'statistics' => 'Statistik',
            'online'     => 'Online',
            'today'      => 'Heute',
            'week'       => 'Woche',
            'month'      => 'Monat',
            'year'       => 'Jahr',
            'total'      => 'Gesamt',
            'max'        => 'Max.',
        ],
        'enEN' => [
            'statistics' => 'Statistics',
            'online'     => 'online',
            'today'      => 'today',
            'week'       => 'week',
            'month'      => 'month',
            'year'       => 'year',
            'total'      => 'total',
            'max'        => 'max.',
        ],
    ];

    /* ------------------------------------------------------------------ */

    public function __construct()
    {
        parent::__construct();
        date_default_timezone_set('Europe/Berlin');

        $this->datafile_rel = basename(__DIR__) . '/' . self::DATA_FILE;
        $this->datafile_abs = PLUGIN_DIR_REL . $this->datafile_rel;

        $this->initLocale();
        $this->loadLanguageFile();
        $this->loadAdminLanguageFile();

        // Datei bei Bedarf anlegen – leeres JSON-Objekt
        if (!file_exists($this->datafile_abs)) {
            file_put_contents($this->datafile_abs, '{}', LOCK_EX);
        }

        // 1) Erstinstallation: Salt automatisch anlegen
        $curSalt = (string)$this->settings->get('hash_salt');
        if ($curSalt === '' || !preg_match('/^[0-9a-f]{32}$/i', $curSalt)) {
            $this->writeNewSalt();
        }

        // 2) Regeneration über gespeicherte Einstellung (nicht $_POST!)
        $regen = (string)$this->settings->get('regen_salt');
        if ($regen !== '' && preg_match('/^neu$/i', $regen)) {
            $this->writeNewSalt();
            if (method_exists($this->settings, 'set')) {
                $this->settings->set('regen_salt', '');
                if (method_exists($this->settings, 'save')) {
                    $this->settings->save();
                }
            }
        }
    }

    /* ------------------------------------------------------------------ */

    public function getContent($value)
    {
        // Salt laden + validieren
        $hashSalt = (string)$this->settings->get('hash_salt');
        if ($hashSalt === '' || !preg_match('/^[0-9a-f]{32}$/i', $hashSalt)) {
            return $this->errorMsg('Der Geheime Salt ist nicht korrekt konfiguriert! Bitte in den Plugin-Einstellungen speichern/neu erzeugen.');
        }

        $now   = time();
        $rawIp = $this->getClientIp();

        // IP-Hash mit Salt
        $clientHash = hash('sha256', $rawIp . $hashSalt);
        $hostname   = @gethostbyaddr($rawIp) ?: '';

        /* ---------- Einstellungen -------------------------------------- */
        $ipLock           = (int)$this->settings->get('ip_lock_sec');
        $userOnlineWindow = (int)$this->settings->get('ip_user_online_sec');

        /* ---------- Bot-Erkennung -------------------------------------- */
        $countVisitor = true;
        $ua       = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $uaBots   = ['bot', 'spider', 'crawl', 'search'];
        $hostBots = ['amazonaws.com', 'kyivstar.net', 'sovam.net.ua'];

        foreach ($uaBots as $b) {
            if ($ua !== '' && stripos($ua, $b) !== false) { $countVisitor = false; break; }
        }
        if ($countVisitor && $hostname !== '') {
            foreach ($hostBots as $b) {
                if (stripos($hostname, $b) !== false) { $countVisitor = false; break; }
            }
        }

        /* ---------- Cookie-Sperre (SameSite=Lax) ----------------------- */
        setcookie(
            'sc_simplecounter',
            $clientHash,
            [
                'expires'  => $now + $ipLock,
                'path'     => '/',
                'samesite' => 'Lax',
                'secure'   => !empty($_SERVER['HTTPS']),
                'httponly' => true,
            ]
        );

        /* ---------- Daten laden + updaten (mit Lock) ------------------- */
        $result = $this->updateDataLocked(function (array $data) use (
            $now,
            $ipLock,
            $userOnlineWindow,
            $clientHash,
            &$countVisitor
        ): array {

            $data = $this->applyDefaults($data, $now);

            // Wiederholungs-/Lock-Logik
            if ($countVisitor) {
                if (isset($data['clientsFirst'][$clientHash]) &&
                    ($now - $ipLock) < (int)$data['clientsFirst'][$clientHash]) {
                    $countVisitor = false;
                }
            }

            // Zeitstempel aktualisieren
            $data['clients'][$clientHash] = $now;

            // Perioden ggf. resetten (nur im Nicht-Zähl-Fall, wie bisher)
            if (!$countVisitor) {
                $this->resetPeriodsIfNeeded($data);
            }

            // Zählen
            if ($countVisitor) {
                $data['clientsFirst'][$clientHash] = $now;

                $this->incrementPeriod($data, 'day',   'z', 'dayHits');
                $this->incrementPeriod($data, 'week',  'W', 'weekHits');
                $this->incrementPeriod($data, 'month', 'n', 'monthHits');
                $this->incrementPeriod($data, 'year',  'Y', 'yearHits');

                $data['allHits']++;

                if ((int)$data['dayHits'] > (int)$data['maxDayHits']) {
                    $data['maxDayHits']           = (int)$data['dayHits'];
                    $data['maxDayHits_timestamp'] = $now;
                }
            }

            // Online-User / Cleaning
            $online = 0;
            foreach ($data['clients'] as $hash => $ts) {
                $ts = (int)$ts;
                if (($ts + $userOnlineWindow) < $now) {
                    if (($ts + $ipLock) < $now) {
                        unset($data['clients'][$hash], $data['clientsFirst'][$hash]);
                    }
                } else {
                    $online++;
                }
            }

            return [$data, $online];
        });

        if (!is_array($result) || count($result) !== 2) {
            return $this->errorMsg('Datei '.$this->datafile_rel.' nicht verfügbar.');
        }

        [$data, $online] = $result;

        if ($value === 'hidden') {
            return '';
        }

        /* ---------- Ausgabe -------------------------------------------- */
        $listPrefix = trim((string)$this->settings->get('list_prefix'));
        $listPrefix = $listPrefix !== '' ? $listPrefix . ' ' : '';

        $template = trim((string)$this->settings->get('counter_userdefined'));
        if ($template === '') {
            $template = "{IconUndTitel|".$this->t('statistics')."}\n{UserOnline}\n{Heute}\n{Woche}\n{Monat}\n{Jahr}\n{Gesamt}\n{Rekord}";
        }

        $stats = $this->buildStatsModel($data, $online);

        $iconHtml = '<img class="sc_icon" src="'.URL_BASE.PLUGIN_DIR_NAME.'/SimpleCounter/icon.png" alt="" aria-hidden="true">';

        $mode = strtolower(trim((string)$this->settings->get('output_mode')));
        if ($mode === 'inline') {
            $out = $this->renderInline($template, $stats, $iconHtml);
            // Mozilo/Editor: sicherheitshalber alles auf eine Zeile ziehen
            $out = str_replace(['<br />','<br>',"\r","\n"], ' ', $out);
            return '<span class="sc_counter sc_inline">'.$out.'</span>';
        }

        $html = $this->renderList($template, $stats, $iconHtml, $listPrefix);
        $html = str_replace('<br />', '', $html);
        return $html;
    }

    /* ------------------------------------------------------------------ */
    /* Backend-Konfiguration                                              */
    /* ------------------------------------------------------------------ */
    public function getConfig(): array
    {
        if (!$this->adminLangLoaded) {
            $this->initLocale();
            $this->loadAdminLanguageFile();
        }

        return [
            'ip_lock_sec' => [
                'type'        => 'text',
                'description' => $this->ta('cfg_ip_lock_sec', 'Dauer IP-Sperre (Sek. 86 400 = 1 Tag)'),
                'maxlength'   => '6',
                'size'        => '10',
                'regex'       => '/^[0-9]{1,6}$/',
                'regex_error' => $this->ta('cfg_ip_lock_sec_err', 'Nur Ganzzahlen erlaubt (0 – 99 999).'),
            ],
            'ip_user_online_sec' => [
                'type'        => 'text',
                'description' => $this->ta('cfg_ip_user_online_sec', 'Dauer, einen User als online zu führen (Sek.)'),
                'maxlength'   => '6',
                'size'        => '10',
                'regex'       => '/^[0-9]{1,6}$/',
                'regex_error' => $this->ta('cfg_ip_user_online_sec_err', 'Nur Ganzzahlen erlaubt (0 – 99 999).'),
            ],
            'list_prefix' => [
                'type'        => 'text',
                'description' => $this->ta('cfg_list_prefix', 'Listen-Aufzählungszeichen (vor Zahl Blockmodus)'),
                'maxlength'   => '8',
                'size'        => '10',
            ],
            'output_mode' => [
                'type'        => 'text',
                'description' => $this->ta('cfg_output_mode', 'Ausgabe-Modus: list (Blockausgabe) oder inline (Zeilenausgabe)'),
                'maxlength'   => '6',
                'size'        => '10',
                'regex'       => '/^(list|inline)$/',
                'regex_error' => $this->ta('cfg_output_mode_err', 'Bitte "list" oder "inline" eintragen.'),
                'default'     => 'list',
            ],			
            'counter_userdefined' => [
                'type'        => 'textarea',
                'cols'        => '50',
                'rows'        => '10',
                'description' => $this->ta(
                    'cfg_counter_userdefined',
                    'Benutzerdef. Counter-Darstellung: 1) Anzahl und Reihenfolge Zähler im Modus list = Blockmodus; 2) Eingabe Zähler mit freiem Text im Modus inline; Zeilenmodus z.B. für Footer'
                ),
            ],
            'hash_salt' => [
                'type'        => 'text',
                'description' => $this->ta('cfg_hash_salt', 'SALT AKTUELL - nach Plugininstallation automatisch eingetragen.'),
                'maxlength'   => '32',
                'size'        => '35',
                'regex'       => '/^[0-9a-fA-F]{32}$/',
                'regex_error' => $this->ta('cfg_hash_salt_err', 'Ungültiger Salt. Muss ein 32-stelliger Hexadezimal-Code sein.'),
                'default'     => '',
            ],
            'regen_salt' => [
                'type'        => 'text',
                'description' => $this->ta('cfg_regen_salt', 'SALT NEU erzeugen. Dazu NEU eintippen + Speichern (Diskettensymbol). UNBEDINGT HILFE BEACHTEN!'),
                'maxlength'   => '3',
                'size'        => '8',
                'regex'       => '/^(|[Nn][Ee][Uu])$/',
                'regex_error' => $this->ta('cfg_regen_salt_err', 'Bitte NEU eingeben oder Feld leer lassen.'),
                'default'     => '',
            ],
        ];
    }

    /* ------------------------------------------------------------------ */
    /* Plugin-Info                                                        */
    /* ------------------------------------------------------------------ */
    public function getInfo(): array
    {
        if (!$this->adminLangLoaded) {
            $this->initLocale();
            $this->loadAdminLanguageFile();
        }

        // Minimaler Fallback (der lange Text steht in admin_language_deDE.txt unter info_html)
        $infoHtml = $this->ta('info_html', '<p>Informationstext nicht gefunden. Bitte admin_language_deDE.txt prüfen (Key: info_html).</p>');

        return [
            '<b>SimpleCounter-2</b> 3.0.4',
            '2.0 / 3.0',
            $infoHtml,
            'hausl78, A. Fleischmann, G. Fischer',
            'https://www.mozilo.de/media',
            [
                '{SimpleCounter}'        => $this->ta('insert_counter', 'Counter einfügen'),
                '{SimpleCounter|hidden}' => $this->ta('insert_hidden', 'nur zählen, nichts anzeigen'),
            ],
        ];
    }

    /* ------------------------------------------------------------------ */
    /* Hilfs-Methoden                                                     */
    /* ------------------------------------------------------------------ */

    /** Locale bestimmen (deDE/enEN) */
    private function initLocale(): void
    {
        $cmsLang = isset($_SESSION['cmslanguage']) ? (string)$_SESSION['cmslanguage'] : '';
        $prefix  = strtolower(substr($cmsLang, 0, 2));
        $this->locale = ($prefix === 'de' || $prefix === '') ? 'deDE' : 'enEN';
    }

    /** Frontend-Sprachdatei laden: sprachen/language_deDE.txt usw. */
    private function loadLanguageFile(): void
    {
        if ($this->langLoaded) {
            return;
        }

        $dir  = rtrim(__DIR__, '/\\') . DIRECTORY_SEPARATOR . self::LANG_DIR;
        $file = $dir . DIRECTORY_SEPARATOR . 'language_' . $this->locale . '.txt';

        // Fallback: wenn nicht vorhanden, versuche deDE
        if (!is_file($file)) {
            $file = $dir . DIRECTORY_SEPARATOR . 'cms_language_deDE.txt';
        }

        $this->i18n = $this->readKeyValueFile($file);
        $this->langLoaded = true;
    }

    /** Admin-Sprachdatei laden: sprachen/admin_language_deDE.txt usw. */
    private function loadAdminLanguageFile(): void
    {
        if ($this->adminLangLoaded) {
            return;
        }

        $dir  = rtrim(__DIR__, '/\\') . DIRECTORY_SEPARATOR . self::LANG_DIR;
        $file = $dir . DIRECTORY_SEPARATOR . 'admin_language_' . $this->locale . '.txt';

        // Fallback: wenn nicht vorhanden, versuche deDE
        if (!is_file($file)) {
            $file = $dir . DIRECTORY_SEPARATOR . 'admin_language_deDE.txt';
        }

        $this->i18nAdmin = $this->readKeyValueFile($file);
        $this->adminLangLoaded = true;
    }

    /** Key=Value Datei lesen (Kommentare # ;, Leerzeilen erlaubt) */
    private function readKeyValueFile(string $file): array
    {
        if (!is_file($file) || !is_readable($file)) {
            return [];
        }

        $lines = file($file, FILE_IGNORE_NEW_LINES);
        if (!is_array($lines)) {
            return [];
        }

        $map = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '' || $line[0] === '#' || $line[0] === ';') {
                continue;
            }
            $pos = strpos($line, '=');
            if ($pos === false) {
                continue;
            }
            $k = trim(substr($line, 0, $pos));
            $v = trim(substr($line, $pos + 1));
            if ($k !== '') {
                $map[$k] = $v;
            }
        }
        return $map;
    }

    /** Frontend-Übersetzung holen (robust: Sprachdatei -> Default-Map -> Key) */
    private function t(string $key, ?string $fallback = null): string
    {
        if (!$this->langLoaded) {
            $this->initLocale();
            $this->loadLanguageFile();
        }
        if (isset($this->i18n[$key]) && $this->i18n[$key] !== '') {
            return (string)$this->i18n[$key];
        }

        $loc = isset(self::DEFAULT_I18N[$this->locale]) ? $this->locale : 'deDE';
        if ($fallback === null && isset(self::DEFAULT_I18N[$loc][$key])) {
            return (string)self::DEFAULT_I18N[$loc][$key];
        }
        return $fallback ?? $key;
    }

    /** Admin-Übersetzung holen */
    private function ta(string $key, string $fallback): string
    {
        if (!$this->adminLangLoaded) {
            $this->initLocale();
            $this->loadAdminLanguageFile();
        }
        if (isset($this->i18nAdmin[$key]) && $this->i18nAdmin[$key] !== '') {
            return (string)$this->i18nAdmin[$key];
        }
        return $fallback;
    }

    /** Neuen Salt erzeugen und speichern */
    private function writeNewSalt(): void
    {
        $new = $this->generateSalt();
        if (method_exists($this->settings, 'set')) {
            $this->settings->set('hash_salt', $new);
            if (method_exists($this->settings, 'save')) {
                $this->settings->save();
            }
        }
    }

    /** Periodenzähler erhöhen */
    private function incrementPeriod(array &$d, string $key, string $fmt, string $hits): void
    {
        if ((string)$d[$key] === date($fmt)) {
            $d[$hits] = (int)$d[$hits] + 1;
        } else {
            $d[$key]  = date($fmt);
            $d[$hits] = 1;
        }
    }

    /** Formatierte Zahl */
    private function n(int $n): string
    {
        return number_format($n, 0, '', '.');
    }

    /** Fehlermeldung */
    private function errorMsg(string $msg): string
    {
        return '<span class="deadlink">SimpleCounter-Fehler: '.$msg.'</span>';
    }

    /** Client-IP */
    private function getClientIp(): string
    {
        foreach (['HTTP_CLIENT_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR'] as $k) {
            if (!empty($_SERVER[$k])) {
                if (strpos((string)$_SERVER[$k], ',') !== false) {
                    return trim(explode(',', (string)$_SERVER[$k])[0]);
                }
                return (string)$_SERVER[$k];
            }
        }
        return '0.0.0.0';
    }

    /** 128-Bit Salt erzeugen */
    private function generateSalt(): string
    {
        try {
            return bin2hex(random_bytes(16));
        } catch (\Throwable $e) {
            if (function_exists('openssl_random_pseudo_bytes')) {
                $strong = false;
                $bytes  = openssl_random_pseudo_bytes(16, $strong);
                if ($bytes && $strong === true) {
                    return bin2hex($bytes);
                }
            }
            throw new \RuntimeException('Konnte keinen sicheren Zufallswert erzeugen');
        }
    }

    /* ---------------------- Data / Lock I/O --------------------------- */

    /**
     * Locked Update: lädt JSON, ruft $fn($data) auf und speichert anschließend wieder.
     * $fn muss entweder $data zurückgeben oder [$data, $extra] (z.B. online).
     */
    private function updateDataLocked(callable $fn)
    {
        $fh = @fopen($this->datafile_abs, 'c+');
        if (!$fh || !@flock($fh, LOCK_EX)) {
            if (is_resource($fh)) { @fclose($fh); }
            return null;
        }

        $size = @filesize($this->datafile_abs);
        $json = $size ? (string)@fread($fh, $size) : '{}';
        $data = json_decode($json, true);
        if (!is_array($data)) {
            $data = [];
        }

        $out = $fn($data);

        // Ergebnis normalisieren: $out = $data oder [$data, $extra]
        $dataToSave = $out;
        if (is_array($out) && count($out) === 2 && is_array($out[0])) {
            $dataToSave = $out[0];
        }

        @rewind($fh);
        @ftruncate($fh, 0);
        @fwrite($fh, json_encode($dataToSave));
        @fflush($fh);

        @flock($fh, LOCK_UN);
        @fclose($fh);

        return $out;
    }

    /** Defaults sicherstellen */
    private function applyDefaults(array $data, int $now): array
    {
        $data += [
            'day'   => date('z'),
            'week'  => date('W'),
            'month' => date('n'),
            'year'  => date('Y'),
            'dayHits'   => 0,
            'weekHits'  => 0,
            'monthHits' => 0,
            'yearHits'  => 0,
            'allHits'   => 0,
            'maxDayHits'           => 0,
            'maxDayHits_timestamp' => $now,
            'clients'              => [],
            'clientsFirst'         => [],
        ];
        if (!is_array($data['clients'])) { $data['clients'] = []; }
        if (!is_array($data['clientsFirst'])) { $data['clientsFirst'] = []; }
        return $data;
    }

    /**
     * Reset-Logik nur im Nicht-Zähl-Fall (entspricht deinem bisherigen Code):
     * - Bei neuem Tag: dayHits=0 und clientsFirst=[]
     * - Bei neuer Woche/Monat/Jahr: Hits=0
     */
    private function resetPeriodsIfNeeded(array &$data): void
    {
        if ((string)$data['day'] !== date('z')) {
            $data['day']          = date('z');
            $data['dayHits']      = 0;
            $data['clientsFirst'] = [];
        }
        if ((string)$data['week'] !== date('W')) {
            $data['week']     = date('W');
            $data['weekHits'] = 0;
        }
        if ((string)$data['month'] !== date('n')) {
            $data['month']     = date('n');
            $data['monthHits'] = 0;
        }
        if ((string)$data['year'] !== date('Y')) {
            $data['year']     = date('Y');
            $data['yearHits'] = 0;
        }
    }

    /* ---------------------- Rendering -------------------------------- */

    /** Stat-Model für Rendering */
    private function buildStatsModel(array $data, int $online): array
    {
        $recordDate = date('d.m.Y', (int)$data['maxDayHits_timestamp']);
        return [
            'online' => $online,
            'day'    => (int)$data['dayHits'],
            'week'   => (int)$data['weekHits'],
            'month'  => (int)$data['monthHits'],
            'year'   => (int)$data['yearHits'],
            'total'  => (int)$data['allHits'],
            'record' => (int)$data['maxDayHits'],
            'record_date' => $recordDate,
        ];
    }

    private function renderInline(string $template, array $s, string $iconHtml): string
    {
        $map = $this->buildInlinePlaceholderMap($s);

        // Titel / Icon im Inline-Modus
        $template = preg_replace(
            '#\{IconUndTitel\|(.*)\}#U',
            '<span class="sc_title">'.$iconHtml.' $1</span>',
            $template
        );
        $template = preg_replace(
            '#\{Titel(?:\|(.*))?\}#U',
            '<span class="sc_title">$1</span>',
            $template
        );

        return strtr($template, $map);
    }

    private function renderList(string $template, array $s, string $iconHtml, string $listPrefix): string
    {
        $map = $this->buildListPlaceholderMap($s, $listPrefix);

        // Titel / Icon im Listenmodus
        $template = preg_replace(
            '#\{IconUndTitel\|(.*)\}#U',
            '<li class="sc_title">'.$iconHtml.' $1</li>',
            $template
        );
        $template = preg_replace(
            '#\{Titel(?:\|(.*))?\}#U',
            '<li class="sc_title">$1</li>',
            $template
        );

        $html = '<ul class="sc_counter">'.strtr($template, $map).'</ul>';
        return $html;
    }

    private function buildInlinePlaceholderMap(array $s): array
    {
        $onlineLabel = $this->t('online');
        $todayLabel  = $this->t('today');
        $weekLabel   = $this->t('week');
        $monthLabel  = $this->t('month');
        $yearLabel   = $this->t('year');
        $totalLabel  = $this->t('total');
        $maxLabel    = $this->t('max');

        $recordText = $maxLabel.': '.$this->n((int)$s['record']).' ('.$s['record_date'].')';

        // Unterstützt sowohl alte als auch neue Namensvarianten (robust)
        return [
            '{UserOnline}' => $s['online'].' '.$onlineLabel,
            '{Heute}'      => $this->n((int)$s['day']).' '.$todayLabel,
            '{Woche}'      => $this->n((int)$s['week']).' '.$weekLabel,
            '{Monat}'      => $this->n((int)$s['month']).' '.$monthLabel,
            '{Jahr}'       => $this->n((int)$s['year']).' '.$yearLabel,
            '{Gesamt}'     => $this->n((int)$s['total']).' '.$totalLabel,
            '{Rekord}'     => $recordText,

            '{UserOnlineWert}' => (string)$s['online'],
            '{HeuteWert}'      => $this->n((int)$s['day']),
            '{WocheWert}'      => $this->n((int)$s['week']),
            '{MonatWert}'      => $this->n((int)$s['month']),
            '{JahrWert}'       => $this->n((int)$s['year']),
            '{GesamtWert}'     => $this->n((int)$s['total']),
            '{RekordWert}'     => $this->n((int)$s['record']),
            '{RekordDatum}'    => (string)$s['record_date'],

            '{OnlineLabel}' => $onlineLabel,
            '{HeuteLabel}'  => $todayLabel,
            '{WocheLabel}'  => $weekLabel,
            '{MonatLabel}'  => $monthLabel,
            '{JahrLabel}'   => $yearLabel,
            '{GesamtLabel}' => $totalLabel,
            '{RekordLabel}' => $maxLabel,

            // “Toleranz”-Aliasse (falls jemand das mal so nutzt)
            '{Total}'       => $this->n((int)$s['total']).' '.$totalLabel,
            '{TotalWert}'   => $this->n((int)$s['total']),
            '{TotalLabel}'  => $totalLabel,
        ];
    }

    private function buildListPlaceholderMap(array $s, string $listPrefix): array
    {
        $onlineLabel = $this->t('online');
        $todayLabel  = $this->t('today');
        $weekLabel   = $this->t('week');
        $monthLabel  = $this->t('month');
        $yearLabel   = $this->t('year');
        $totalLabel  = $this->t('total');
        $maxLabel    = $this->t('max');

        $recordText = $maxLabel.': '.$this->n((int)$s['record']).' ('.$s['record_date'].')';

        return [
            '{UserOnline}' => '<li class="sc_online">'.$listPrefix.$s['online'].' '.$onlineLabel.'</li>',
            '{Heute}'      => '<li class="sc_today">'.$listPrefix.$this->n((int)$s['day']).' '.$todayLabel.'</li>',
            '{Woche}'      => '<li class="sc_week">'.$listPrefix.$this->n((int)$s['week']).' '.$weekLabel.'</li>',
            '{Monat}'      => '<li class="sc_month">'.$listPrefix.$this->n((int)$s['month']).' '.$monthLabel.'</li>',
            '{Jahr}'       => '<li class="sc_year">'.$listPrefix.$this->n((int)$s['year']).' '.$yearLabel.'</li>',
            '{Gesamt}'     => '<li class="sc_total">'.$listPrefix.$this->n((int)$s['total']).' '.$totalLabel.'</li>',
            '{Rekord}'     => '<li class="sc_record">'.$recordText.'</li>',
        ];
    }
}
